﻿FUNCTION Confirm-AdequateDiskSpace {
<#
	.SYNOPSIS
		Verifies adequate disk space is available
	
	.DESCRIPTION
		Verifies adequate disk space is available to continue with the current process.
	
	.PARAMETER WorkAreaObject
		ProjectWise Work Area / Folder object to get document size summation for.
	
	.PARAMETER LocalDrive
		Local drive letter to verify disk space. Must be drive letter and colon (d:).
	
	.EXAMPLE
		PS C:\> Confirm-AdequateDiskSpace -WorkAreaObject $WorkAreaObject -LocalDrive 'd:'
	
	.NOTES
		This cmdlet is intended to be used with ProjectWise Work Area / Folder objects to ensure there
		is enough disk space available on the local drive. This will prevent processes from failing due
		to running out of resources.
#>
	
	[CmdletBinding(PositionalBinding = $true)]
	[OutputType([boolean])]
	param (
		[ValidateNotNullOrEmpty()]
		[ValidateScript({ Get-PWFolders -FolderPath $_.FullPath -Justone })]		
        [Parameter(Mandatory = $true,
				   ValueFromPipeline = $true,
				   Position = 0,
				   HelpMessage = 'ProjectWise Work Area object to report on.')]
		[psobject]$WorkAreaObject,
		[ValidateNotNullOrEmpty()]
		[ValidatePattern('^.:$')]
		[Parameter(Mandatory = $true,
				   Position = 1,
				   HelpMessage = 'Local drive to verify space for.')]
		[string]$LocalDrive
	) # end param...
	
	BEGIN {
		$CmdletName = $MyInvocation.InvocationName
		$StartTime = Get-Date
		Write-Verbose "[BEGIN] $StartTime - Entering '$CmdletName' Function..."
	} # end BEGIN...
	
	PROCESS {
		
		try {
			
			#region CREATE TEMPORARY SAVED SEARCH
			
			# Name of temporary saved search to be created.
			$SearchName = 'AllDocs_PSTemp'
			
			$Splat_SavedSearch = @{
				OwnerProject = $WorkAreaObject
				SearchName   = $SearchName
				SearchFolder = $WorkAreaObject.FullPath
				SearchSubFolders = $true
				ReplaceExisting = $true
			}
			$SavedSearch = Add-PWSavedSearch @Splat_SavedSearch
			
			$test = Get-PWSavedSearches -InputFolder $OwnerProject
			if (-not ($test | Where-Object Name -eq $SearchName)) {
				Write-Warning -Message "Failed to add temporary saved search."
				throw
			}
			
			#endregion CREATE TEMPORARY SAVED SEARCH
			
			#region GET PROJECTWISE WORKAREA DOCUMENT SIZE
			
			# Get summation of documents filesize based on temporary saved search.
			$pwDocsData = Get-PWDocumentsBySearch -SearchName $SavedSearch | 
            Measure-Object FileSize -Sum
			[long]$WorkAreaFileSize = $pwDocsData.Sum
			
			Remove-PWSavedSearch -InputFolder $WorkAreaObject -SearchName $SearchName
			
			#endregion GET PROJECTWISE WORKAREA DOCUMENT SIZE
			
			# Get available disk space for local drive provided.
			[long]$diskFreeSpace = Get-CimInstance -Class CIM_LogicalDisk | 
            Where-Object DeviceID -eq $LocalDrive | 
            Select-Object -ExpandProperty Freespace
			
            <# Double WorkAreaFileSize and subtract from diskFreeSpace
            to determine if there is enough diskspace to complete process. #>
			
			if ($diskFreeSpace - ($WorkAreaFileSize * 2) -lt 0) {
				Write-Warning -Message "Not enough disk space to run process. Required: $([math]::Round(($WorkAreaFileSize * 2)/1GB, 2)) GB  Available: $([math]::Round($diskFreeSpace/1GB, 2)) GB"
				Write-Output $false
			} else {
				Write-Output $true
			}
			
		} catch {
			Write-Warning -Message "[PROCESS] $($Error[0])"
		}
		
	} # end PROCESS...
	
	END {
		$EndTime = Get-Date
		Write-Verbose "[END] It took $($EndTime - $StartTime) to complete the process."
		Write-Verbose "[END] $EndTime - Exiting '$CmdletName'..."
	} # end END...
} # end FUNCTION Confirm-AdequateDiskSpace
Export-ModuleMember -Function Confirm-AdequateDiskSpace

